#include <stdint.h>

#define RCU_APB2EN 0x40021018
#define GPIOA_CTL0 0x40010800
#define GPIOA_OCTL 0x4001080C

#define TIMER_FREQ ((uint32_t)8000000/4)
#define TIMER_CTRL_ADDR 0xd1000000
#define TIMER_MTIME 0x0
#define TIMER_MTIMECMP 0x8

#define ECLIC_ADDR_BASE 0xd2000000
#define ECLIC_CLICINTIE_OFFSET 0x1001
#define ECLIC_CLICINTATTR_OFFSET 0x1002

extern uint32_t _vector_base[];

void gpio_init()
{
    // enable GPIOA clock
    *(uint32_t *)RCU_APB2EN |= (uint32_t)1 << 2;

    // set PA1 as output push pull
    *(uint32_t *)GPIOA_CTL0 = *(uint32_t *)GPIOA_CTL0 & ((uint32_t)0xffffff0f) |
                              (uint32_t)1 << 4;

    // PA1 output low
    *(uint32_t *)GPIOA_OCTL &= (uint32_t)0xfffffffd;
}

void timer_init()
{
    *(volatile uint32_t *)(TIMER_CTRL_ADDR + TIMER_MTIME + 4) = 0;
    *(volatile uint32_t *)(TIMER_CTRL_ADDR + TIMER_MTIME) = 0;
    *(volatile uint32_t *)(TIMER_CTRL_ADDR + TIMER_MTIMECMP + 4) = 0;
    *(volatile uint32_t *)(TIMER_CTRL_ADDR + TIMER_MTIMECMP) = TIMER_FREQ;
}

int main()
{
    gpio_init();
    timer_init();

    uint32_t base = (uint32_t)_vector_base;
    // ECLIC mode
    asm volatile("csrw mtvec, %0" :: "r"(base | 3));
    asm volatile("csrw mtvt, %0" :: "r"(base));

    // vectored mode
    *(volatile uint32_t *)(ECLIC_ADDR_BASE + ECLIC_CLICINTATTR_OFFSET +
            7 * 4) |= 0x1; 
    // level trigger mode
    *(volatile uint32_t *)(ECLIC_ADDR_BASE + ECLIC_CLICINTATTR_OFFSET +
            7 * 4) &= 0xfffffff9; 
    // enable timer interrupt
    *(volatile uint32_t *)(ECLIC_ADDR_BASE + ECLIC_CLICINTIE_OFFSET +
            7 * 4) |= 0x1;
    // enable global interrupt
    asm volatile("csrs mstatus, %0" :: "r"(1 << 3));

    while(1);

    return 0;
}

void timer_handler()
{
    // reset mtime
    *(volatile uint32_t *)(TIMER_CTRL_ADDR + TIMER_MTIME + 4) = 0;
    *(volatile uint32_t *)(TIMER_CTRL_ADDR + TIMER_MTIME) = 0;
    
    // toggle PA1 output
    if(*(uint32_t *)GPIOA_OCTL & (uint32_t)0b10){
        *(uint32_t *)GPIOA_OCTL &= (uint32_t)0xfffffffd;
    } else{
        *(uint32_t *)GPIOA_OCTL |= (uint32_t)0b10;
    }
}

