#include "ftl.h"
#include "spi_flash.h"
#include <stdint.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>

struct ftl_manager ftlm;
uint32_t index_serial_buf[INDEX_BLK_NUM];
__attribute__((aligned(4))) uint8_t read_sec_buf[READ_BUF_SIZE];
__attribute__((aligned(4))) uint8_t read_sec_buf2[READ_BUF_SIZE];
__attribute__((aligned(4))) uint8_t write_sec_buf[WRITE_BUF_SIZE];
__attribute__((aligned(4))) uint8_t l2p_buf[L2P_BUF_SIZE];
__attribute__((aligned(4))) uint8_t l2p_buf2[L2P_BUF_SIZE];

void flash_chip_erase()
{
#if FTL_DEBUG
    printf("doing chip erase...\r\n");
#endif
    spi_flash_erase_chip();
#if FTL_DEBUG
    printf("erase complete\r\n");
#endif
}

void flash_erase(uint16_t blk)
{
    spi_flash_erase_block_64kb((int)(BLK_TO_ADDR(blk)));
}

void flash_program(uint8_t *buf, uint16_t psec, uint16_t len)
{
    spi_flash_write_buffer(buf, (uint32_t)(SEC_TO_ADDR(psec)), (uint32_t)len);
}

void flash_read(uint8_t *buf, uint16_t psec, uint16_t off, uint16_t len)
{
    // printf("%x %x\r\n", (uint32_t)(SEC_TO_ADDR(psec) + off), len);
    spi_flash_read_buffer(buf, (uint32_t)(SEC_TO_ADDR(psec) + off), (uint32_t)len);
}

uint8_t flash_blk_is_free(uint16_t blk)
{
    uint8_t temp;
    flash_read(&temp, BLK_TO_SEC(blk), 0, 1);
    if(temp == 0xff){
        return 1;
    }
    return 0;
}

uint8_t flash_psec_is_free(uint16_t psec)
{
    flash_read(read_sec_buf2, psec, 0, SEC_PER_BLK);
    for(size_t i = 0; i < SECTOR_SIZE; i++){
        if(*(uint8_t *)(read_sec_buf2 + i) != 0xff){
            return 0;
        }
    }
    return 1;
}

/*
* how much data sector could be use for data storaging
*/
uint32_t get_capacity()
{
    return (DATA_BLK_NUM - RES_DATA_BLK) * (SEC_PER_BLK - DATA_BLK_HEADER_SEC - DATA_BLK_TAIL_SEC);
}

static void get_index_serial()
{
    for(size_t i = 0; i < INDEX_BLK_NUM; i++){
        flash_read((uint8_t *)(&index_serial_buf[i]),
                   BLK_TO_SEC(i), 4, sizeof(uint32_t));
    }
}

/*
* increment index serial
*/
static uint32_t inc_index_serial(uint32_t serial)
{
    FTL_ASSERT(serial != (uint32_t)-1);
    uint32_t res = serial + 1;
    if(res == (uint32_t)-1)
        return 0;
    return res;
}

/*
 * compare index serial a and b, if a newer than b, return 1, else return false
*/
static uint8_t index_serial_cmp(uint32_t a, uint32_t b)
{
    FTL_ASSERT(a != (uint32_t)-1);
    FTL_ASSERT(b != (uint32_t)-1);
    if(a == b)
        return 0;
    if(a - b <= INDEX_BLK_NUM)
        return 1;
    return 0;
}

static uint8_t has_value(uint16_t value, uint16_t *buf, size_t len)
{
    for(size_t i = 0; i < len; i++){
        if(value == buf[i])
            return 1;
    }
    return 0;
}

static void get_l2p_item(l2p_item_t *item, uint8_t *buf, size_t off)
{
    memcpy((void *)item, (void *)buf + sizeof(l2p_item_t) * off, sizeof(l2p_item_t));
}

static void write_l2p_item(l2p_item_t *item, uint8_t *buf, size_t off)
{
    memcpy((void *)buf + sizeof(l2p_item_t) * off, (void *)item, sizeof(l2p_item_t));
}

static uint8_t l2p_buf_is_free()
{
    for(size_t i = 0; i < L2P_BUF_SIZE; i++){
        if(l2p_buf[i] != 0xff)
            return 0;
    }
    return 1;
}

uint16_t get_psec(uint16_t lsec)
{
    l2p_item_t item;
    if(!l2p_buf_is_free()){
        for(size_t i = 0; i < L2P_BUF_SIZE / L2P_ITEM_SIZE; i++){
            get_l2p_item(&item, l2p_buf,
                L2P_BUF_SIZE / L2P_ITEM_SIZE - (i + 1));
            if(item.lsec == lsec)
                return item.psec;
        }
    }
    uint16_t temp[INDEX_BLK_NUM];
    memset((void *)temp, 0xff, sizeof(temp));
    for(size_t i = 0; i < INDEX_BLK_NUM; i++){
        uint32_t max = (uint32_t)-1;
        for(size_t j = 0; j < INDEX_BLK_NUM; j++){
            if(index_serial_buf[j] == (uint32_t)-1 || has_value(j, temp, INDEX_BLK_NUM))
                continue;
            if(max ==(uint32_t)-1 || index_serial_cmp(index_serial_buf[j], max)){
                max = index_serial_buf[j];
                temp[i] = j;
            }
        }

        if(temp[i] == (uint16_t)-1)
            break;
        for(size_t j = 0; j < SEC_PER_BLK - INDEX_BLK_RES_SEC; j++){
            uint16_t sec = BLK_TO_SEC(temp[i]) + SEC_PER_BLK - (j + 1);
            if(flash_psec_is_free(sec))
                continue;
            flash_read(l2p_buf2, sec, 0, L2P_BUF_SIZE);
            for(size_t k = 0; k < L2P_BUF_SIZE / L2P_ITEM_SIZE; k++){
                get_l2p_item(&item, l2p_buf2,
                    L2P_BUF_SIZE / L2P_ITEM_SIZE - (k + 1));
                if(item.lsec == lsec)
                    return item.psec;
            }
        }
    }
    return (uint16_t)-1;
}

uint8_t read_lsec(uint8_t *buf, uint16_t lsec, uint16_t len)
{
    FTL_ASSERT(len <= SECTOR_SIZE);
    uint16_t psec = get_psec(lsec);
    if(psec == (uint16_t)-1)
        return (uint8_t)-1;
#if FTL_DEBUG
    printf("read: lsec: %x, psec: %x\r\n", lsec, psec);
#endif
    flash_read(buf, psec, 0, len);
    return 0;
}

uint8_t read_lsec_seq(uint8_t *buf, uint16_t lsec, uint16_t sec_num)
{
    for(size_t i = 0; i < sec_num; i++){
        if(read_lsec(buf + SECTOR_SIZE * i, lsec + i, SECTOR_SIZE) == 0xff){
            return 0xff;
        }
    }
    return 0;
}

/*
* just for index compacting
*/
static uint8_t has_l2p_item(uint16_t *blk, uint16_t len, uint16_t lsec)
{
    l2p_item_t item;
    for(size_t i = 0; i < L2P_BUF_SIZE / L2P_ITEM_SIZE; i++){
        get_l2p_item(&item, l2p_buf2, i);
        if(item.lsec == (uint16_t)-1)
            break;
        if(item.lsec == lsec)
            return 1;
    }
    for(size_t i = 0; i < len; i++){
        if(blk[i] == (uint16_t)-1)
            break;
        for(size_t j = INDEX_BLK_RES_SEC; j < SEC_PER_BLK; j++){
            if(flash_psec_is_free(BLK_TO_SEC(blk[i]) + j))
                break;
            flash_read(read_sec_buf2, BLK_TO_SEC(blk[i]) + j, 0, SECTOR_SIZE);
            for(size_t k = 0; k < READ_BUF_SIZE / L2P_ITEM_SIZE; k++){
                get_l2p_item(&item, read_sec_buf2, k);
                if(item.lsec == (uint16_t)-1)
                    return 0;
                if(item.lsec == lsec)
                    return 1;
            }
        }
    }
    return 0;
}

static uint16_t get_free_index_blk()
{
    for(uint16_t i = 0; i < INDEX_BLK_NUM; i++){
        if(index_serial_buf[i] == (uint32_t)-1){
            return i;
        }
    }
    return (uint16_t)-1;
}

static void prepare_index_blk(uint16_t blk, uint32_t serial)
{
    uint8_t buf[8] = {0};
    memcpy((void *)buf + 4, (void *)&serial, 4);
    flash_program(buf, BLK_TO_SEC(blk), 8);
}

void index_compact()
{
#if FTL_DEBUG
    printf("do indices compacting\r\n");
#endif
    uint16_t temp[INDEX_BLK_NUM];
    uint32_t max_serial = (uint32_t)-1;
    memset((void *)temp, 0xff, sizeof(temp));
    for(size_t i = 0; i < INDEX_BLK_NUM; i++){
        uint32_t max = (uint32_t)-1;
        for(size_t j = 0; j < INDEX_BLK_NUM; j++){
            if(index_serial_buf[j] == (uint32_t)-1 || has_value(j, temp, INDEX_BLK_NUM))
                continue;
            if(max ==(uint32_t)-1 || index_serial_cmp(index_serial_buf[j], max)){
                max = index_serial_buf[j];
                temp[i] = j;
            }
        }
        if(max_serial == (uint32_t)-1)
            max_serial = max;
    }
    if(temp[0] == (uint16_t)-1)
        return;

    uint16_t new_blk[INDEX_BLK_NUM];
    size_t item_moved = 0;
    size_t sec_index = INDEX_BLK_RES_SEC;
    size_t blk_index = 0;

    memset((void *)new_blk, 0xff, sizeof(new_blk));
    new_blk[0] = get_free_index_blk();
    FTL_ASSERT(new_blk[0] != (uint16_t)-1);
    max_serial = inc_index_serial(max_serial);
    prepare_index_blk(new_blk[0], max_serial);
    memset((void*)l2p_buf2, 0xff, L2P_BUF_SIZE);

    for(size_t i = 0; i < INDEX_BLK_NUM; i++){
        if(temp[i] == (uint16_t)-1)
            break;

        for(size_t j = 0; j < SEC_PER_BLK - INDEX_BLK_RES_SEC; j++){
            uint16_t sec = BLK_TO_SEC(temp[i]) + SEC_PER_BLK - (j + 1);
            if(flash_psec_is_free(sec))
                continue;
            flash_read(read_sec_buf, sec, 0, READ_BUF_SIZE);
            for(size_t k = 0; k < READ_BUF_SIZE / L2P_ITEM_SIZE; k++){
                l2p_item_t item;
                get_l2p_item(&item, read_sec_buf,
                    READ_BUF_SIZE / L2P_ITEM_SIZE - (k + 1));
                if(item.lsec == (uint16_t)-1)
                    continue;
                if(!has_l2p_item(new_blk, INDEX_BLK_NUM, item.lsec)){
                    write_l2p_item(&item, l2p_buf2,
                        item_moved % (L2P_BUF_SIZE / L2P_ITEM_SIZE));
                    item_moved++;
                    if(!(item_moved % (L2P_BUF_SIZE / L2P_ITEM_SIZE))){
                        flash_program(l2p_buf2,
                            BLK_TO_SEC(new_blk[blk_index]) + sec_index,
                            L2P_BUF_SIZE);
                        sec_index++;
                        if(sec_index == SEC_PER_BLK){
                            blk_index++;
                            new_blk[blk_index] = get_free_index_blk();
                            FTL_ASSERT(new_blk[blk_index] != (uint16_t)-1);
                            max_serial = inc_index_serial(max_serial);
                            prepare_index_blk(new_blk[blk_index], max_serial);
                            sec_index = INDEX_BLK_RES_SEC;
                        }
                        memset((void*)l2p_buf2, 0xff, L2P_BUF_SIZE);
                    }
                }
            }
        }
        flash_erase(temp[i]);
    }
    if(item_moved % (L2P_BUF_SIZE / L2P_ITEM_SIZE)){
        flash_program(l2p_buf2,
            BLK_TO_SEC(new_blk[blk_index]) + sec_index,
            L2P_BUF_SIZE);
        sec_index++;
    }
    ftlm.index_sec_off = sec_index - 1;
    ftlm.index_item_off = 0;
    get_index_serial();
}

__attribute__((aligned(4))) uint8_t data_blk_mask[DATA_BLK_NUM / 2];

static uint8_t get_data_blk_mask_value(uint16_t blk)
{
    FTL_ASSERT(blk >= INDEX_BLK_NUM && blk < BLOCK_COUNT);
    uint16_t index = (blk - INDEX_BLK_NUM) / 2;
    if(blk & 0x1){
        return data_blk_mask[index] & 0xf;
    }
    return data_blk_mask[index] >> 4;
}

static void set_data_blk_mask_value(uint16_t blk, uint8_t value)
{
    FTL_ASSERT(blk >= INDEX_BLK_NUM && blk < BLOCK_COUNT);
    uint16_t index = (blk - INDEX_BLK_NUM) / 2;
    if(blk & 0x1){
        data_blk_mask[index] &= 0xf0;
        data_blk_mask[index] |= value & 0x0f;
    } else{
        data_blk_mask[index] &= 0x0f;
        data_blk_mask[index] |= value << 4;
    }
}

static void get_data_blk_mask()
{
    ftlm.free_data_blk_num = 0;
    uint8_t temp;
    for(size_t i = INDEX_BLK_NUM; i < BLOCK_COUNT; i++){
        flash_read(&temp, BLK_TO_SEC(i), 0, 1);
        if(temp == 0xff){
            set_data_blk_mask_value(i, DATA_BLK_MASK_FREE);
            ftlm.free_data_blk_num++;
            continue;
        }
        flash_read(&temp, BLK_TO_SEC(i) + SEC_PER_BLK - 1, 0,1);
        if(!temp){
            set_data_blk_mask_value(i, DATA_BLK_MASK_FULL);
        } else{
            set_data_blk_mask_value(i, DATA_BLK_MASK_USED);
        }
    }
}

static uint16_t get_free_data_blk()
{
    for(uint16_t i = INDEX_BLK_NUM; i < BLOCK_COUNT; i++){
        if(get_data_blk_mask_value(i) == DATA_BLK_MASK_FREE){
            return i;
        }
    }
    return (uint16_t)-1;
}

static uint16_t get_used_data_blk()
{
    for(uint16_t i = INDEX_BLK_NUM; i < BLOCK_COUNT; i++){
        if(get_data_blk_mask_value(i) == DATA_BLK_MASK_USED){
            return i;
        }
    }
    return (uint16_t)-1;
}

static uint16_t get_writable_data_blk()
{
    uint16_t mask;
    for(uint16_t i = INDEX_BLK_NUM; i < BLOCK_COUNT; i++){
        mask = get_data_blk_mask_value(i);
        if(mask == DATA_BLK_MASK_FREE || mask == DATA_BLK_MASK_USED){
            return i;
        }
    }
    return (uint16_t)-1;
}

/*
* one bit per sector, garbage sector is marked as 1
*/
uint8_t sec_mask[DATA_BLK_NUM * SEC_PER_BLK / 8];

static void set_sec_mask_value(uint16_t sec, uint8_t value)
{
    uint16_t index = (sec - INDEX_BLK_NUM * SEC_PER_BLK) / 8;
    if(value){
        sec_mask[index] |= 1 << (7 - (sec & 0x7));
    } else{
        sec_mask[index] &= ~(1 << (7 - (sec & 0x7)));
    }
}

static uint8_t get_sec_mask_value(uint16_t sec)
{
    uint16_t index = (sec - INDEX_BLK_NUM * SEC_PER_BLK) / 8;
    if(sec_mask[index] & (1 << (7 - (sec & 0x7))))
        return 1;
    return 0;
}

/*
* get logical sector, need to do a index compacting first
*/
static uint16_t get_lsec(uint16_t psec)
{
    for(uint16_t i = 0; i < INDEX_BLK_NUM; i++){
        if(flash_blk_is_free(i))
            continue;
        for(uint16_t j = INDEX_BLK_RES_SEC; j < SEC_PER_BLK; j++){
            uint16_t sec = SEC_PER_BLK * i + j;
            if(flash_psec_is_free(sec))
                break;
            flash_read(read_sec_buf, sec, 0, READ_BUF_SIZE);
            for(size_t k = 0; k < SECTOR_SIZE / L2P_ITEM_SIZE; k++){
                l2p_item_t item;
                get_l2p_item(&item, read_sec_buf, k);
                if(item.psec == psec)
                    return item.lsec;
            }
        }
    }
    return (uint16_t)-1;
}

/*
* check if physical sector sec is a garbage sector,
* need to do a index compacting first
*/
static uint8_t is_garbage(uint16_t psec)
{
    if(get_lsec(psec) != (uint16_t)-1)
        return 0;
    return 1;
}

/*
* get garbage sector bit map, need to do a index
* compacting first
*/
static void get_sec_mask()
{
    memset((void *)sec_mask, 0, sizeof(sec_mask));
    for(size_t i = INDEX_BLK_NUM; i < BLOCK_COUNT; i++){
        for(size_t j = DATA_BLK_HEADER_SEC; j < SEC_PER_BLK - DATA_BLK_TAIL_SEC; j++){
            uint16_t sec = SEC_PER_BLK * i + j;
            set_sec_mask_value(sec, is_garbage(sec));
        }
    }
}

static uint32_t get_max_index_serial()
{
    uint32_t max = (uint32_t)-1;
    for(size_t i = 0; i < INDEX_BLK_NUM; i++){
        if(index_serial_buf[i] == (uint32_t)-1)
            continue;
        if(max == (uint32_t)-1 || index_serial_cmp(index_serial_buf[i], max)){
            max = index_serial_buf[i];
        }
    }
    return max;
}

static uint16_t get_latest_index_blk()
{
    uint32_t serial = get_max_index_serial();
    for(uint16_t i = 0; i < INDEX_BLK_NUM; i++){
        if(serial == index_serial_buf[i])
            return i;
    }
    return (uint16_t)-1; // never reach here
}

static uint16_t free_index_blk_total()
{
    uint16_t ret = 0;
    for(uint16_t i = 0; i < INDEX_BLK_NUM; i++){
        if(index_serial_buf[i] == (uint32_t)-1)
            ret++;
    }
    return ret;
}

static uint8_t flush_l2p_buf()
{
    if(l2p_buf_is_free())
        return 0;
    if(ftlm.index_sec_off == SEC_PER_BLK - 1){
        if(free_index_blk_total() <= RES_INDEX_BLK)
            return 0xff;
        uint16_t new_blk = get_free_index_blk();
        uint32_t serial = get_max_index_serial();
        prepare_index_blk(new_blk, serial == (uint32_t)-1 ? 0 : inc_index_serial(serial));
        get_index_serial();
        ftlm.index_sec_off = INDEX_BLK_RES_SEC - 1;
    }
    ftlm.index_sec_off++;
    uint16_t blk = get_latest_index_blk();
    if(blk == (uint16_t)-1)
        blk = 0;
    flash_program(l2p_buf, BLK_TO_SEC(blk) + ftlm.index_sec_off, L2P_BUF_SIZE);
    memset((void *)l2p_buf, 0xff, sizeof(l2p_buf));
    ftlm.index_item_off = 0;
    return 0;
}

static uint8_t add_l2p_item(l2p_item_t *item)
{
    if(ftlm.index_item_off == L2P_BUF_SIZE / L2P_ITEM_SIZE){
        if(flush_l2p_buf() == 0xff){
            index_compact();
            if(flush_l2p_buf() == 0xff)
                return 0xff;
        }
    }
    write_l2p_item(item, l2p_buf, ftlm.index_item_off);
    ftlm.index_item_off++;
    return 0;
}

/*
* check a used block's sectors are all valid, partial valid or all garbage
*/
static uint8_t get_blk_status(uint16_t blk)
{
    FTL_ASSERT(blk >= INDEX_BLK_NUM);
    FTL_ASSERT(!flash_blk_is_free(blk));

    uint16_t count = 0;
    for(uint16_t i = BLK_TO_SEC(blk) + DATA_BLK_HEADER_SEC;
      i < BLK_TO_SEC(blk) + SEC_PER_BLK - DATA_BLK_TAIL_SEC; i++){
        if(get_sec_mask_value(i)){
            count++;
        }
    }
    if(!count)
        return ALL_VALID;
    if(count == SEC_PER_BLK - DATA_BLK_HEADER_SEC - DATA_BLK_TAIL_SEC)
        return ALL_GARBAGE;
    return PARTIAL_VALID;
}

static void prepare_data_blk(uint16_t blk)
{
    uint8_t buf[4] = {0};
    flash_program(buf, BLK_TO_SEC(blk), 4);
}

static void commit_data_blk(uint16_t blk)
{
    uint8_t buf[4] = {0};
    flash_program(buf, BLK_TO_SEC(blk) + SEC_PER_BLK - DATA_BLK_TAIL_SEC, 4);
}

uint8_t gc(uint16_t count)
{
    if(ftlm.free_data_blk_num < RES_DATA_BLK)
        return 0xff;
    if(flush_l2p_buf() == 0xff)
        return 0xff;
    index_compact();
    get_sec_mask();
    get_data_blk_mask();
    uint16_t new_blk = get_free_data_blk();
    if(new_blk == (uint16_t)-1)
        return 0xff;
    l2p_item_t item;
    uint8_t blk_mask;
    uint8_t blk_status;
    uint16_t psec;
    uint16_t new_blk_sec_off = DATA_BLK_HEADER_SEC;
    uint16_t blk_free = RES_DATA_BLK;
    for(uint16_t i = INDEX_BLK_NUM; i < BLOCK_COUNT; i++){
        blk_mask = get_data_blk_mask_value(i);
        if(blk_mask != DATA_BLK_MASK_FULL && blk_mask != DATA_BLK_MASK_USED)
            continue;
        blk_status = get_blk_status(i);
        if(blk_status == ALL_VALID){
            continue;
        } else if(blk_status == PARTIAL_VALID){
            for(uint16_t j = DATA_BLK_HEADER_SEC; j < SEC_PER_BLK - DATA_BLK_TAIL_SEC; j++){
                psec = BLK_TO_SEC(i) + j;
                if(!get_sec_mask_value(psec)){
                    if(new_blk_sec_off == DATA_BLK_HEADER_SEC){
                        new_blk = get_free_data_blk();
                        if(new_blk == (uint16_t)-1)
                            return 0xff;
                        prepare_data_blk(new_blk);
                        set_data_blk_mask_value(new_blk, DATA_BLK_MASK_TODO);
                        FTL_ASSERT(blk_free >= RES_DATA_BLK);
                        blk_free--;
                        FTL_ASSERT(ftlm.free_data_blk_num >= RES_DATA_BLK);
                        ftlm.free_data_blk_num--;
                    }
                    item.lsec = get_lsec(psec);
                    item.psec = BLK_TO_SEC(new_blk) + new_blk_sec_off;
                    if(add_l2p_item(&item) == 0xff)
                        return 0xff;
                    flash_read(write_sec_buf, psec, 0, WRITE_BUF_SIZE);
                    flash_program(write_sec_buf,
                        BLK_TO_SEC(new_blk) + new_blk_sec_off, WRITE_BUF_SIZE);
                    new_blk_sec_off++;
                    if(new_blk_sec_off == SEC_PER_BLK - DATA_BLK_TAIL_SEC){
                        new_blk_sec_off = DATA_BLK_HEADER_SEC;
                        commit_data_blk(new_blk);
                        set_data_blk_mask_value(new_blk, DATA_BLK_MASK_FULL);
                    }
                }
            }
        }
        flash_erase(i);
        blk_free++;
        ftlm.free_data_blk_num++;
        set_data_blk_mask_value(i, DATA_BLK_MASK_FREE);
        if(count && blk_free >= count + RES_DATA_BLK){
            break;
        }
    }
    if(get_data_blk_mask_value(new_blk) == DATA_BLK_MASK_TODO)
        set_data_blk_mask_value(new_blk, DATA_BLK_MASK_USED);
    if(blk_free >= RES_DATA_BLK){
        return 0;
    }
    return 0xff;
}

void ftl_init()
{
    spi_flash_init();
    get_index_serial();
    uint16_t blk = get_latest_index_blk();
    if(!blk && index_serial_buf[0] == (uint32_t)-1){
        ftlm.index_sec_off = SEC_PER_BLK - 1;
    } else{
        for(uint16_t i = 0; i < SEC_PER_BLK - INDEX_BLK_RES_SEC; i++){
            uint16_t sec = BLK_TO_SEC(blk) + SEC_PER_BLK - i - 1;
            if(!flash_psec_is_free(sec)){
                ftlm.index_sec_off = sec % SEC_PER_BLK;
                break;
            }
        }
    }
    ftlm.index_item_off = 0;
    get_data_blk_mask(); // will calculate free data block number
    ftlm.data_blk_off = (uint16_t)-1;
    ftlm.data_sec_off = SEC_PER_BLK - DATA_BLK_TAIL_SEC - 1;
    memset((void *)l2p_buf, 0xff, L2P_BUF_SIZE);
}

uint8_t write_lsec(uint8_t *buf, uint16_t lsec, uint16_t len)
{
    FTL_ASSERT(len <= SECTOR_SIZE);
    uint16_t psec;
    if(ftlm.data_sec_off == SEC_PER_BLK - DATA_BLK_TAIL_SEC - 1){
        if(ftlm.data_blk_off != (uint16_t)-1)
            commit_data_blk(ftlm.data_blk_off);
        get_data_blk_mask();
        ftlm.data_blk_off = get_writable_data_blk();
        if(ftlm.data_blk_off == (uint16_t)-1)
            return 0xff;
        if(get_data_blk_mask_value(ftlm.data_blk_off) == DATA_BLK_MASK_FREE){
            do{
                if(ftlm.free_data_blk_num <= RES_DATA_BLK){
                    ftlm.data_blk_off = get_used_data_blk();
                    if(ftlm.data_blk_off == (uint16_t)-1){
                        gc(1);
                        ftlm.data_blk_off = get_used_data_blk();
                        if(ftlm.data_blk_off != (uint16_t)-1)
                            break;
                        ftlm.data_blk_off = get_free_data_blk();
                        if(ftlm.free_data_blk_num <= RES_DATA_BLK)
                            return 0xff;
                    } else{
                        break;
                    }
                }
                ftlm.free_data_blk_num--;
                prepare_data_blk(ftlm.data_blk_off);
                ftlm.data_sec_off = DATA_BLK_HEADER_SEC - 1;
            }while(0);
        } 
        if(get_data_blk_mask_value(ftlm.data_blk_off) == DATA_BLK_MASK_USED){
            if(flush_l2p_buf() == 0xff){
                index_compact();
                if(flush_l2p_buf() == 0xff)
                    return 0xff;
            }
            index_compact();
            ftlm.data_sec_off = (uint16_t)-1;
            for(uint16_t i = SEC_PER_BLK - DATA_BLK_TAIL_SEC - 1; i >= DATA_BLK_HEADER_SEC; i--){
                psec = BLK_TO_SEC(ftlm.data_blk_off) + i;
                if(!(get_lsec(psec) == (uint16_t)-1 && flash_psec_is_free(psec))){
                    ftlm.data_sec_off = i;
                    break;
                }
            }
            FTL_ASSERT(ftlm.data_sec_off < SEC_PER_BLK - DATA_BLK_TAIL_SEC - 1);
            FTL_ASSERT(ftlm.data_sec_off >= DATA_BLK_HEADER_SEC);
        }
    }
    ftlm.data_sec_off++;
    psec = BLK_TO_SEC(ftlm.data_blk_off) + ftlm.data_sec_off;
    l2p_item_t item;
    item.lsec = lsec;
    item.psec = psec;
    if(add_l2p_item(&item) == 0xff)
        return 0xff;
#if FTL_DEBUG
    printf("write: lsec: %x, psec: %x\r\n", lsec, psec);
#endif
    flash_program(buf, psec, len);
    return 0;
}

uint8_t write_lsec_seq(uint8_t *buf, uint16_t lsec, uint16_t sec_num)
{
    for(size_t i = 0; i < sec_num; i++){
        if(write_lsec(buf + SECTOR_SIZE * i, lsec + i, SECTOR_SIZE) == 0xff){
            return 0xff;
        }
    }
    if(flush_l2p_buf() == 0xff){
        index_compact();
        if(flush_l2p_buf() == 0xff)
            return 0xff;
    }
    return 0;
}
